<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Mail\QuoteRequestMail;
use App\Models\QuoteRequest;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;
use Throwable;

class QuoteRequestController extends Controller
{
    /**
     * CORS headers helper for API responses
     */
    private function corsHeaders(): array
    {
        return [
            'Access-Control-Allow-Origin'  => '*',
            'Access-Control-Allow-Methods' => 'GET, POST, OPTIONS',
            'Access-Control-Allow-Headers' => 'Content-Type, Authorization',
        ];
    }

    /**
     * Insert a new quote request and send an email immediately (not queued).
     */
    public function insert(Request $request)
    {
        try {
            // 1) Validate incoming JSON
            $data = $request->validate([
                'first_name'     => 'required|string|max:100',
                'surname'        => 'required|string|max:100',
                'phone'          => 'required|string|max:50',
                'email'          => 'required|email',
                'city'           => 'required|string|max:100',
                'state'          => 'required|string|max:100',
                'postcode'       => 'required|string|max:20',
                'message'        => 'nullable|string',
                'product_name'   => 'required|string|max:150',
                'product_url'    => 'nullable|url|max:500',
                'product_price'  => 'nullable|string|max:50',
                'product_sku'    => 'required|string|max:100',
                'product_qty'    => 'required|int|max:100',
                'product_configuration' => 'nullable|json',
            ]);

            // 2) Token + expiry (48 hours as mentioned in email)
            $data['token']            = Str::random(40);
            $data['token_expires_at'] = Carbon::now()->addHours(48);

            // 3) Persist
            $quote = QuoteRequest::create($data);

            // 4) Notify (synchronous send)
            try {
                // Use default mailer; change to ->mailer('smtp') if you prefer
				
				// $link = $quote->product_url.'?SSID='.$quote->token;
				
				// Mail::mailer(config('mail.default'))->send('emails.quotes.request', ['quote' => $quote, 'link' => $link], function ($message) use ($data) {
				// 	$message->to($data['email'])
				// 			->subject('Your Quote Request');
				// });
				$link = $quote->product_url . '?SSID=' . urlencode($quote->token);

                // Parse product_configuration (string, array, or object)
                $configs = null;
                if (!empty($quote->product_configuration)) {
                    if (is_array($quote->product_configuration)) {
                        $configs = $quote->product_configuration;
                    } elseif (is_object($quote->product_configuration)) {
                        $configs = json_decode(json_encode($quote->product_configuration), true);
                    } else {
                        $decoded = json_decode((string) $quote->product_configuration, true);
                        $configs = (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) ? $decoded : null;
                    }
                }

                // Build query parameters
                $params = [
                    'quantity' => $quote->product_qty,
                ];

                // Add product configurations
                if (is_array($configs)) {
                    foreach ($configs as $config) {
                        if (!is_array($config)) {
                            continue;
                        }
                        $name = isset($config['name']) ? trim((string)$config['name']) : null;
                        $value = $config['value'] ?? ($config['label'] ?? null);

                        if ($name && $value !== null) {
                            $paramName = preg_replace('/\s+/', '_', $name);
                            $params[$paramName] = $value;
                        }
                    }
                }

                // Append params to link
                if (!empty($params)) {
                    $link .= '&' . http_build_query($params);
                }

                // Send email
                Mail::mailer(config('mail.default'))->send(
                    'emails.quotes.request',
                    [
                        'quote' => $quote,
                        'link'  => $link,
                    ],
                    function ($message) use ($data) {
                        $message->to($data['email'])
                                ->subject('Your Quote Request');
                    }
                );

				Mail::mailer(config('mail.default'))->send('emails.quotes.admin_request', ['quote' => $quote, 'link' => $link], function ($message) {
					$message->to('prayash@omnionline.com.au')
                            ->cc('common.mritu@gmail.com')
					//$message->to('jayesh@mjailab.com')
							->subject('Admin: Customer Quote Request');
				});


            } catch (Throwable $e) {
                // Log full detail to storage/logs/laravel.log
                Log::error('QuoteRequest mail failed', [
                    'exception' => $e, // Monolog will include full trace
                    'message'   => $e->getMessage(),
                    'trace'     => $e->getTraceAsString(),
                ]);

                // Return a 500 with details (trace only if APP_DEBUG=true)
                return response()
                    ->json([
                        'success' => false,
                        'message' => 'Failed to send quote request email.',
                        'error'   => $e->getMessage(),
                        'trace'   => config('app.debug') ? $e->getTraceAsString() : null,
                    ], 500)
                    ->withHeaders($this->corsHeaders());
            }

            // 5) Success
            return response()
                ->json([
                    'success' => true,
                    'id'      => $quote->id,
                    'token'   => $quote->token,
                    'expires' => $quote->token_expires_at?->toISOString(),
                ], 201)
                ->withHeaders($this->corsHeaders());

        } catch (ValidationException $ve) {
            return response()
                ->json([
                    'success' => false,
                    'message' => 'Validation failed.',
                    'errors'  => $ve->errors(),
                ], 422)
                ->withHeaders($this->corsHeaders());

        } catch (Throwable $e) {
            Log::error('QuoteRequest insert failed', [
                'exception' => $e,
                'message'   => $e->getMessage(),
                'trace'     => $e->getTraceAsString(),
            ]);

            return response()
                ->json([
                    'success' => false,
                    'message' => 'Unexpected server error.',
                    'error'   => $e->getMessage(),
                    'trace'   => config('app.debug') ? $e->getTraceAsString() : null,
                ], 500)
                ->withHeaders($this->corsHeaders());
        }
    }

    /**
     * Validate a token for viewing a quote (checks 48h expiry).
     */
    public function validateToken(string $token)
    {
        $quote = QuoteRequest::where('token', $token)->first();

        $ok = $quote && Carbon::now()->lessThanOrEqualTo($quote->token_expires_at);

        return response()
            ->json(['success' => $ok ? true : false], 200)
            ->withHeaders($this->corsHeaders());
    }
}
